import { API_ENDPOINTS } from "@/lib/apiEndpoints";
import { Metadata } from "next";
import HandwrittenAssignmentDetailClient from "./HandwrittenAssignmentDetailClient";
import { buildAssignmentDetailMeta } from "@/lib/seo/assignmentMeta";

export const dynamic = "force-dynamic";

async function fetchHandwrittenAssignmentData(courseCode: string, assignmentCode: string) {
    try {
        const res = await API_ENDPOINTS.ASSIGNMENTS.DETAIL(courseCode, assignmentCode);
        if (!res.ok) throw new Error(`HTTP ${res.status}`);
        const data = await res.json();
        return data.assignment;
    } catch (err: any) {
        console.error("❌ Failed to load handwritten assignment details", err);
        return null;
    }
}

export async function generateMetadata({ params }: { params: Promise<{ slug: string; code: string }> }): Promise<Metadata> {
    const resolvedParams = await params;
    const assignment = await fetchHandwrittenAssignmentData(resolvedParams.slug, resolvedParams.code);
    if (!assignment) return { title: "Handwritten Assignment" };

    const meta = buildAssignmentDetailMeta(assignment);
    return {
        ...meta,
        alternates: {
            canonical: `/assignments/handwritten/${resolvedParams.slug}/${resolvedParams.code}`,
        }
    };
}

export default async function HandwrittenProductPage({ params }: { params: Promise<{ slug: string; code: string }> }) {
    const resolvedParams = await params;
    const assignment = await fetchHandwrittenAssignmentData(resolvedParams.slug, resolvedParams.code);

    if (!assignment) {
        return <div className="text-danger text-center py-5">Failed to load assignment details.</div>;
    }

    return (
        <HandwrittenAssignmentDetailClient
            productData={assignment}
            courseCode={resolvedParams.slug}
            assignmentCode={resolvedParams.code}
        />
    );
}
